USE AdventureWorks2014;
SET ANSI_NULLS ON;
SET QUOTED_IDENTIFIER ON;
SET NOCOUNT ON;
GO
-- Create IndexMetrics schema, if it does not already exist.
IF NOT EXISTS (SELECT * FROM sys.schemas WHERE schemas.name = 'IndexMetrics')
BEGIN
	DECLARE @Schema_Creation_TSQL NVARCHAR(MAX);
	SELECT @Schema_Creation_TSQL = 'CREATE SCHEMA IndexMetrics;';
	EXEC sp_executesql @Schema_Creation_TSQL;
END
GO

CREATE TABLE IndexMetrics.Missing_Index_Details
(	Missing_Index_Details_Id INT NOT NULL IDENTITY(1,1) CONSTRAINT PK_Missing_Index_Details PRIMARY KEY CLUSTERED,
	Missing_Index_Details_Create_Datetime DATETIME NOT NULL,
	[Database_Name] SYSNAME NOT NULL,
	[Schema_Name] SYSNAME NOT NULL,
	Table_Name VARCHAR(256) NOT NULL,
	Equality_Columns VARCHAR(MAX) NULL,
	Inequality_Columns VARCHAR(MAX) NULL,
	Include_Columns VARCHAR(MAX) NULL,
	Last_User_Seek DATETIME NOT NULL,
	Avg_Total_User_Cost FLOAT NOT NULL,
	Avg_User_Impact FLOAT NOT NULL,
	User_Seeks BIGINT NOT NULL,
	Index_Group_Handle INT NOT NULL,
	Index_Handle INT NOT NULL
);
GO
CREATE NONCLUSTERED INDEX IX_Missing_Index_Details_lastUserSeek ON IndexMetrics.Missing_Index_Details(Last_User_Seek);
GO

CREATE TABLE IndexMetrics.Missing_Index_Summary
(	Missing_Index_Summary_Id INT NOT NULL IDENTITY(1,1) CONSTRAINT PK_Missing_Index_Summary PRIMARY KEY CLUSTERED,
	[Database_Name] SYSNAME NOT NULL,
	[Schema_Name] SYSNAME NOT NULL,
	Table_Name VARCHAR(256) NOT NULL,
	Equality_Columns VARCHAR(MAX) NOT NULL,
	Inequality_Columns VARCHAR(MAX) NOT NULL,
	Include_Columns VARCHAR(MAX) NOT NULL,
	First_Index_Suggestion_Time DATETIME NOT NULL,
	Last_User_Seek DATETIME NOT NULL,
	Avg_Total_User_Cost FLOAT NOT NULL,
	Avg_User_Impact FLOAT NOT NULL,
	User_Seeks BIGINT NOT NULL,
	User_Seeks_Last_Update BIGINT NOT NULL
);
GO
CREATE NONCLUSTERED INDEX IX_Missing_Index_Summary_lastUserSeek ON IndexMetrics.Missing_Index_Summary(Last_User_Seek);
CREATE NONCLUSTERED INDEX IX_Missing_Index_Summary_databaseName_tableName ON IndexMetrics.Missing_Index_Summary([Database_Name], Table_Name);
GO

IF EXISTS (SELECT * FROM sys.procedures WHERE procedures.name = 'Populate_Missing_Index_Data')
BEGIN
	DROP PROCEDURE IndexMetrics.Populate_Missing_Index_Data
END
GO
/*
	This stored procedure populates the missing index tables with a point-in-time and summary view of data from SQL Server's missing index DMVs.
*/
CREATE PROCEDURE IndexMetrics.Populate_Missing_Index_Data
	@Retention_Period_for_Detail_Data_Days SMALLINT = 30,
	@Delete_All_Summary_Data BIT = 0,
	@Aggregate_All_Database_Data BIT = 0 -- If set to 1, all collected data will use "ALL" instad of the database name.  This is useful in multi-tenant environments with many databases containing the same schema/usage patterns.
	-- This bit should be set and kept with the same value long-term.  Switching this back and forth will lead to confusing data points.  If changing this, clearing out (or aggregating) all existing data first is recommended.
AS
BEGIN
	SET NOCOUNT ON;

	DELETE Missing_Index_Details
	FROM IndexMetrics.Missing_Index_Details
	WHERE Missing_Index_Details.Last_User_Seek < DATEADD(DAY, -1 * @Retention_Period_for_Detail_Data_Days, CURRENT_TIMESTAMP);

	IF @Delete_All_Summary_Data = 1
	BEGIN
		TRUNCATE TABLE IndexMetrics.Missing_Index_Summary;
	END

	DECLARE @Last_Seek_Time DATETIME;
	SELECT
		@Last_Seek_Time = MAX(Missing_Index_Details.Last_User_Seek)
	FROM IndexMetrics.Missing_Index_Details;
	IF @Last_Seek_Time IS NULL
	BEGIN
		SELECT @Last_Seek_Time = DATEADD(WEEK, -1, CURRENT_TIMESTAMP);
	END

	-- Generate a database list so that we collect data from all databases on the server
	DECLARE @Database_List TABLE
		(	[Database_Name] SYSNAME NOT NULL,
			Is_Processed BIT NOT NULL);

	DECLARE @Sql_Command NVARCHAR(MAX);
	DECLARE @Current_Database_Name SYSNAME;

	INSERT INTO @Database_List
		([Database_Name], Is_Processed)
	SELECT
		databases.name AS [Database_Name],
		0 AS Is_Processed
	FROM sys.databases
	WHERE databases.name NOT IN ('master', 'msdb', 'model', 'tempdb', 'ReportServerTempDB', 'ReportServer')
	AND databases.state_desc = 'ONLINE';

	CREATE TABLE #Missing_Index_Details
	(	Missing_Index_Details_Id INT IDENTITY(1,1) NOT NULL PRIMARY KEY CLUSTERED,
		[Database_Name] SYSNAME,
		[Schema_Name] SYSNAME,
		Table_Name SYSNAME,
		Equality_Columns VARCHAR(MAX),
		Inequality_Columns VARCHAR(MAX),
		Include_Columns VARCHAR(MAX),
		Last_User_Seek DATETIME,
		Avg_Total_User_Cost FLOAT,
		Avg_User_Impact FLOAT,
		User_Seeks BIGINT,
		Index_Group_Handle INT,
		Index_Handle INT);

	-- Loop through each database and collect missing index stats for each.
	WHILE EXISTS (SELECT * FROM @Database_List Database_List WHERE Database_List.Is_Processed = 0)
	BEGIN
		SELECT TOP 1
			@Current_Database_Name = Database_List.[Database_Name]
		FROM @Database_List Database_List
		WHERE Database_List.Is_Processed = 0;

		SELECT @Sql_Command = '
		USE [' + @Current_Database_Name + '];

		INSERT INTO #Missing_Index_Details
			([Database_Name], [Schema_Name], Table_Name, Equality_Columns, Inequality_Columns, Include_Columns, Last_User_Seek,
			 Avg_Total_User_Cost, Avg_User_Impact, User_Seeks, Index_Group_Handle, Index_Handle)
		SELECT
			databases.name AS [Database_Name],
			schemas.name AS [Schema_Name],
			tables.name AS Table_Name,
			dm_db_missing_index_details.Equality_Columns,
			dm_db_missing_index_details.Inequality_Columns,
			dm_db_missing_index_details.Included_Columns AS Include_Columns,
			dm_db_missing_index_group_stats.Last_User_Seek,
			dm_db_missing_index_group_stats.Avg_Total_User_Cost,
			dm_db_missing_index_group_stats.Avg_User_Impact,
			dm_db_missing_index_group_stats.User_Seeks,
			dm_db_missing_index_groups.Index_Group_Handle,
			dm_db_missing_index_groups.Index_Handle
		FROM sys.dm_db_missing_index_groups
		INNER JOIN sys.dm_db_missing_index_group_stats
		ON dm_db_missing_index_group_stats.group_handle = dm_db_missing_index_groups.index_group_handle
		INNER JOIN sys.dm_db_missing_index_details
		ON dm_db_missing_index_groups.index_handle = dm_db_missing_index_details.index_handle
		INNER JOIN sys.databases
		ON databases.database_id = dm_db_missing_index_details.database_id
		INNER JOIN sys.tables
		ON tables.[object_id] = dm_db_missing_index_details.[object_id]
		INNER JOIN sys.schemas
		ON schemas.[schema_id] = tables.[schema_id]
		WHERE databases.name = ''' + @Current_Database_Name + '''
		AND dm_db_missing_index_group_stats.Last_User_Seek > ''' + CAST(@Last_Seek_Time AS NVARCHAR(MAX)) + ''';';

		EXEC sp_executesql @Sql_Command;

		UPDATE Database_List
			SET Is_Processed = 1
		FROM @Database_List Database_List
		WHERE [Database_Name] = @Current_Database_Name;
	END

	INSERT INTO IndexMetrics.Missing_Index_Details
		(Missing_Index_Details_Create_Datetime, [Database_Name], [Schema_Name], Table_Name, Equality_Columns, Inequality_Columns, Include_Columns,
		 Last_User_Seek, Avg_Total_User_Cost, Avg_User_Impact, User_Seeks, Index_Group_Handle, Index_Handle)
	SELECT DISTINCT
		CURRENT_TIMESTAMP AS Missing_Index_Details_Create_Datetime,
		[Database_Name],
		[Schema_Name],
		Table_Name,
		Equality_Columns,
		Inequality_Columns,
		Include_Columns,
		Last_User_Seek,
		Avg_Total_User_Cost,
		Avg_User_Impact,
		User_Seeks,
		Index_Group_Handle,
		Index_Handle
	FROM #Missing_Index_Details;

	IF @Aggregate_All_Database_Data = 0
	BEGIN
		MERGE INTO IndexMetrics.Missing_Index_Summary AS Index_Summary_Target
		USING (SELECT [Database_Name], [Schema_Name], Table_Name, ISNULL(Equality_Columns, '') AS Equality_Columns, ISNULL(Inequality_Columns, '') AS Inequality_Columns, ISNULL(Include_Columns, '') AS Include_Columns,
					  MAX(Last_User_Seek) AS Last_User_Seek, AVG(Avg_Total_User_Cost) AS Avg_Total_User_Cost, AVG(Avg_User_Impact) AS Avg_User_Impact, SUM(User_Seeks) AS User_Seeks
					  FROM #Missing_Index_Details GROUP BY [Database_Name], [Schema_Name], Table_Name, Equality_Columns, Inequality_Columns, Include_Columns) AS Index_Summary_Source
		ON (Index_Summary_Source.[Database_Name] = Index_Summary_Target.[Database_Name]
			AND Index_Summary_Source.[Schema_Name] = Index_Summary_Target.[Schema_Name]
			AND Index_Summary_Source.Table_Name = Index_Summary_Target.Table_Name
			AND Index_Summary_Source.Equality_Columns = Index_Summary_Target.Equality_Columns
			AND Index_Summary_Source.Inequality_Columns = Index_Summary_Target.Inequality_Columns
			AND Index_Summary_Source.Include_Columns = Index_Summary_Target.Include_Columns)
		WHEN MATCHED
			THEN UPDATE
				SET Last_User_Seek = Index_Summary_Source.Last_User_Seek,
					User_Seeks = CASE 
									  WHEN Index_Summary_Source.User_Seeks = Index_Summary_Target.User_Seeks_Last_Update
											THEN Index_Summary_Target.User_Seeks
							  		  WHEN Index_Summary_Source.User_Seeks >= Index_Summary_Target.User_Seeks
											THEN Index_Summary_Source.User_Seeks + Index_Summary_Target.User_Seeks - Index_Summary_Target.User_Seeks_Last_Update
									  WHEN Index_Summary_Source.User_Seeks < Index_Summary_Target.User_Seeks
									  AND Index_Summary_Source.User_Seeks < Index_Summary_Target.User_Seeks_Last_Update
											THEN Index_Summary_Target.User_Seeks + Index_Summary_Source.User_Seeks
									  WHEN Index_Summary_Source.User_Seeks < Index_Summary_Target.User_Seeks
									  AND Index_Summary_Source.User_Seeks > Index_Summary_Target.User_Seeks_Last_Update
											THEN Index_Summary_Source.User_Seeks + Index_Summary_Target.User_Seeks - Index_Summary_Target.User_Seeks_Last_Update
								 END,
					User_Seeks_Last_Update = Index_Summary_Source.User_Seeks,
					Avg_Total_User_Cost = Index_Summary_Source.Avg_Total_User_Cost,
					Avg_User_Impact = Index_Summary_Source.Avg_User_Impact
		WHEN NOT MATCHED BY TARGET
			THEN INSERT
				VALUES (Index_Summary_Source.[Database_Name], Index_Summary_Source.[Schema_Name], Index_Summary_Source.Table_Name, Index_Summary_Source.Equality_Columns,
						Index_Summary_Source.Inequality_Columns, Index_Summary_Source.Include_Columns, CURRENT_TIMESTAMP, Index_Summary_Source.Last_User_Seek,
						Index_Summary_Source.Avg_Total_User_Cost, Index_Summary_Source.Avg_User_Impact, Index_Summary_Source.User_Seeks, Index_Summary_Source.User_Seeks);
	END
	ELSE
	BEGIN
		MERGE INTO IndexMetrics.Missing_Index_Summary AS Index_Summary_Target
		USING (SELECT 'ALL' AS [Database_Name], [Schema_Name], Table_Name, ISNULL(Equality_Columns, '') AS Equality_Columns, ISNULL(Inequality_Columns, '') AS Inequality_Columns, ISNULL(Include_Columns, '') AS Include_Columns,
					  MAX(Last_User_Seek) AS Last_User_Seek, AVG(Avg_Total_User_Cost) AS Avg_Total_User_Cost, AVG(Avg_User_Impact) AS Avg_User_Impact, SUM(User_Seeks) AS User_Seeks
					  FROM #Missing_Index_Details GROUP BY [Schema_Name], Table_Name, Equality_Columns, Inequality_Columns, Include_Columns) AS Index_Summary_Source
		ON (Index_Summary_Target.[Database_Name] = 'ALL'
			AND Index_Summary_Source.[Schema_Name] = Index_Summary_Target.[Schema_Name]
			AND Index_Summary_Source.Table_Name = Index_Summary_Target.Table_Name
			AND Index_Summary_Source.Equality_Columns = Index_Summary_Target.Equality_Columns
			AND Index_Summary_Source.Inequality_Columns = Index_Summary_Target.Inequality_Columns
			AND Index_Summary_Source.Include_Columns = Index_Summary_Target.Include_Columns)
		WHEN MATCHED
			THEN UPDATE
				SET Last_User_Seek = Index_Summary_Source.Last_User_Seek,
					User_Seeks = CASE 
									  WHEN Index_Summary_Source.User_Seeks = Index_Summary_Target.User_Seeks_Last_Update
											THEN Index_Summary_Target.User_Seeks
							  		  WHEN Index_Summary_Source.User_Seeks >= Index_Summary_Target.User_Seeks
											THEN Index_Summary_Source.User_Seeks + Index_Summary_Target.User_Seeks - Index_Summary_Target.User_Seeks_Last_Update
									  WHEN Index_Summary_Source.User_Seeks < Index_Summary_Target.User_Seeks
									  AND Index_Summary_Source.User_Seeks < Index_Summary_Target.User_Seeks_Last_Update
											THEN Index_Summary_Target.User_Seeks + Index_Summary_Source.User_Seeks
									  WHEN Index_Summary_Source.User_Seeks < Index_Summary_Target.User_Seeks
									  AND Index_Summary_Source.User_Seeks > Index_Summary_Target.User_Seeks_Last_Update
											THEN Index_Summary_Source.User_Seeks + Index_Summary_Target.User_Seeks - Index_Summary_Target.User_Seeks_Last_Update
								 END,
					User_Seeks_Last_Update = Index_Summary_Source.User_Seeks,
					Avg_Total_User_Cost = Index_Summary_Source.Avg_Total_User_Cost,
					Avg_User_Impact = Index_Summary_Source.Avg_User_Impact
		WHEN NOT MATCHED BY TARGET
			THEN INSERT
				VALUES ('ALL', Index_Summary_Source.[Schema_Name], Index_Summary_Source.Table_Name, Index_Summary_Source.Equality_Columns,
						Index_Summary_Source.Inequality_Columns, Index_Summary_Source.Include_Columns, CURRENT_TIMESTAMP, Index_Summary_Source.Last_User_Seek,
						Index_Summary_Source.Avg_Total_User_Cost, Index_Summary_Source.Avg_User_Impact, Index_Summary_Source.User_Seeks, Index_Summary_Source.User_Seeks);
	END

	DROP TABLE #Missing_Index_Details;
END
GO

IF EXISTS (SELECT * FROM sys.views WHERE views.name = 'v_Missing_Index_Details')
BEGIN
	DROP VIEW IndexMetrics.v_Missing_Index_Details;
END
GO

CREATE VIEW IndexMetrics.v_Missing_Index_Details
AS
SELECT
	Missing_Index_Details_Create_Datetime,
	[Database_Name],
	[Schema_Name],
	Table_Name,
	'CREATE NONCLUSTERED INDEX [missing_index_'	+ CONVERT (VARCHAR, Missing_Index_Details.Index_Group_Handle) + '_'	+ CONVERT (VARCHAR, Missing_Index_Details.Index_Handle) + '_' +
		Table_Name + ']' + ' ON ' + Table_Name +
		' (' + ISNULL(Missing_Index_Details.Equality_Columns, '') + CASE WHEN Missing_Index_Details.Equality_Columns IS NOT NULL
																	 AND Missing_Index_Details.Inequality_Columns IS NOT NULL
																		THEN ','
																	 ELSE ''
																END + 
		ISNULL(Missing_Index_Details.Inequality_Columns, '') + ')' + ISNULL(' INCLUDE (' + Missing_Index_Details.Include_Columns + ')', '') AS Index_Creation_Statement,
	Missing_Index_Details.Avg_Total_User_Cost * (Missing_Index_Details.Avg_User_Impact / 100.0) * Missing_Index_Details.User_Seeks AS Improvement_Measure,
	Equality_Columns,
	Inequality_Columns,
	Include_Columns,
	ISNULL(LEN(Missing_Index_Details.Equality_Columns) - LEN(REPLACE(Missing_Index_Details.Equality_Columns, '[', '')), 0) AS Equality_Column_Count,
	ISNULL(LEN(Missing_Index_Details.Inequality_Columns) - LEN(REPLACE(Missing_Index_Details.Inequality_Columns, '[', '')), 0) AS Inequality_Column_Count,
	ISNULL(LEN(Missing_Index_Details.Include_Columns) - LEN(REPLACE(Missing_Index_Details.Include_Columns, '[', '')), 0) AS Included_Column_Count,
	Last_User_Seek,
	User_Seeks,
	Index_Group_Handle,
	Index_Handle
FROM IndexMetrics.Missing_Index_Details;
GO

USE AdventureWorks2014;
GO

IF EXISTS (SELECT * FROM sys.views WHERE views.name = 'v_Missing_Index_Summary')
BEGIN
	DROP VIEW IndexMetrics.v_Missing_Index_Summary;
END
GO

CREATE VIEW IndexMetrics.v_Missing_Index_Summary
AS
SELECT
	[Database_Name],
	[Schema_Name],
	Table_Name,
	'CREATE NONCLUSTERED INDEX [missing_index_'	+ CONVERT (VARCHAR, Missing_Index_Summary.Missing_Index_Summary_Id) + '_' +
		Table_Name + ']' + ' ON ' + Table_Name +
		' (' + Missing_Index_Summary.Equality_Columns + CASE WHEN Missing_Index_Summary.Equality_Columns <> ''
																	 AND Missing_Index_Summary.Inequality_Columns <> ''
																		THEN ','
																	 ELSE ''
																END + 
		Missing_Index_Summary.Inequality_Columns + ')' + CASE WHEN Missing_Index_Summary.Include_Columns = ''
																		THEN ''
																		ELSE ' INCLUDE (' + Missing_Index_Summary.Include_Columns + ')'
																	 END AS Index_Creation_Statement,
	Missing_Index_Summary.Avg_Total_User_Cost * (Missing_Index_Summary.Avg_User_Impact / 100.0) * Missing_Index_Summary.User_Seeks AS Improvement_Measure,
	Missing_Index_Summary.Equality_Columns,
	Missing_Index_Summary.Inequality_Columns,
	Missing_Index_Summary.Include_Columns,
	ISNULL(LEN(Missing_Index_Summary.Equality_Columns) - LEN(REPLACE(Missing_Index_Summary.Equality_Columns, '[', '')), 0) AS Equality_Column_Count,
	ISNULL(LEN(Missing_Index_Summary.Inequality_Columns) - LEN(REPLACE(Missing_Index_Summary.Inequality_Columns, '[', '')), 0) AS Inequality_Column_Count,
	ISNULL(LEN(Missing_Index_Summary.Include_Columns) - LEN(REPLACE(Missing_Index_Summary.Include_Columns, '[', '')), 0) AS Included_Column_Count,
	Missing_Index_Summary.First_Index_Suggestion_Time,
	Missing_Index_Summary.Last_User_Seek,
	Missing_Index_Summary.User_Seeks
FROM IndexMetrics.Missing_Index_Summary;
GO
